"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
var _AuthType = _interopRequireDefault(require("../AuthType"));
var _missing_tenant_error = _interopRequireDefault(require("../../errors/missing_tenant_error"));
var _authentication_error = _interopRequireDefault(require("../../errors/authentication_error"));
var _missing_role_error = _interopRequireDefault(require("../../errors/missing_role_error"));
var _routes = require("./routes");
var _path = _interopRequireDefault(require("path"));
var _AuthManager = require("../../AuthManager");
/*
 *    Copyright 2021 floragunn GmbH
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

class OpenId extends _AuthType.default {
  constructor({
    searchGuardBackend,
    kibanaCore,
    config,
    logger,
    sessionStorageFactory,
    pluginDependencies,
    spacesService
  }) {
    super({
      searchGuardBackend,
      kibanaCore,
      config,
      logger,
      sessionStorageFactory,
      pluginDependencies,
      spacesService
    });

    /**
     * The authType is saved in the auth cookie for later reference
     * @type {string}
     */
    this.type = _AuthManager.AUTH_TYPE_NAMES.OIDC;

    /**
     * If a loginURL is defined, we can skip the auth selector page
     * if the customer only has one auth type enabled.
     * @type {string|null}
     */
    this.loginURL = _routes.OIDC_ROUTES.LOGIN;
  }
  async getRedirectTargetForUnauthenticated(request, error = null, isAJAX = false, sessionCookie = {}) {
    const url = new URL(request.url.href);
    url.pathname = _path.default.posix.join(this.basePath, '/searchguard/login');
    // If we don't have any tenant we need to show the custom error page
    if (error instanceof _missing_tenant_error.default) {
      url.searchParams.set('type', 'missingTenant');
    } else if (error instanceof _missing_role_error.default) {
      url.searchParams.set('type', 'missingRole');
    } else if (error instanceof _authentication_error.default) {
      url.searchParams.set('type', 'authError');
    } else {
      if (!isAJAX) {
        url.searchParams.set('nextUrl', this.getNextUrl(request));
        // Delete sg_tenant because we have it already as a param in the nextUrl
        url.searchParams.delete('sg_tenant');
      }
      url.searchParams.set('configAuthTypeId', sessionCookie.authTypeId || null);
      url.pathname = _path.default.posix.join(this.basePath, this.loginURL);
    }
    return url.pathname + url.search + url.hash;
  }
  setupRoutes() {
    (0, _routes.defineRoutes)({
      authInstance: this,
      kibanaCore: this.kibanaCore,
      kibanaConfig: this.config,
      logger: this.logger,
      debugLog: this.debugLog.bind(this),
      searchGuardBackend: this.searchGuardBackend
    });
  }

  /**
   * Clears the session and logs the user out from the IdP (if we have an endpoint available)
   * @url http://openid.net/specs/openid-connect-session-1_0.html#RPLogout
   * @param context
   * @param request
   * @param response
   * @returns {Promise<*>}
   */
  async logout({
    context = null,
    request,
    response
  }) {
    // @todo Auth error isn't the best message for this. We still
    // get logged out from Kibana, but the IdP logout may fail.
    let redirectURL = `${this.basePath}/customerror?type=authError`;
    const sessionCookie = (await this.sessionStorageFactory.asScoped(request).get()) || {};
    const authHeader = this.getAuthHeader(sessionCookie);
    try {
      const sessionInfo = await this.searchGuardBackend.sessionInfo(authHeader);
      // sso_logout_url doesn't always exist
      redirectURL = sessionInfo.sso_logout_url || this.basePath + '/searchguard/login?type=' + this.type + 'Logout';
    } catch (error) {
      this.logger.error(`OIDC auth logout failed while retrieving the sso_logout_url: ${error.stack}`);
    }

    // Clear the cookie credentials
    await this.clear(request, true);
    return response.ok({
      body: {
        authType: this.type,
        redirectURL
      }
    });
  }
}
exports.default = OpenId;
module.exports = exports.default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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